<?php
/*
Plugin Name: إدارة باقات المواقع
Description: بلجن لإدارة باقات المواقع الفرعية وتفعيل الإضافات والثيمات تلقائيًا مع تصنيفات وأنواع فرعية.
Version: 1.3
Author: أنت
*/

if (!defined('ABSPATH')) {
    exit; // منع الوصول المباشر
}

define('PACKAGES_PLUGIN_DIR', plugin_dir_path(__FILE__));

/**
 * دالة ترجع قائمة أنواع المواقع (التصنيفات + الأنواع الفرعية)
 */
function packages_get_site_types() {
    return [
        'personal' => [
            'label'       => 'مواقع شخصية واحترافية للفرد',
            'description' => 'مواقع مثالية للأفراد، الفريلانسرز، وأصحاب الهويات الشخصية والاحترافية.',
            'examples'    => [
                'موقع شخصي Personal Site',
                'بورتفوليو أعمال (مصمم / مطوّر / مصوّر / فريلانسر)',
                'CV / Resume أونلاين',
                'صفحة Link in Bio (لينك واحد لكل حساباتك)',
                'مدونة شخصية / Blog',
            ],
            'subtypes'    => [
                'personal_site' => 'موقع شخصي Personal Site',
                'portfolio'     => 'بورتفوليو أعمال',
                'cv_online'     => 'CV / Resume أونلاين',
                'link_in_bio'   => 'صفحة Link in Bio',
                'personal_blog' => 'مدونة شخصية / Blog',
            ],
        ],

        'business' => [
            'label'       => 'مواقع شركات وأعمال',
            'description' => 'مخصصة للشركات والوكالات والأعمال التجارية والخدمية.',
            'examples'    => [
                'موقع شركة خدمات (تسويق – برمجة – تصميم – استشارات – صيانة… الخ)',
                'موقع وكالة (Agency)',
                'موقع شركة ناشئة Startup',
                'موقع “من نحن + خدماتنا + أعمالنا + تواصل معنا”',
            ],
            'subtypes'    => [
                'services_company' => 'موقع شركة خدمات',
                'agency'           => 'موقع وكالة (Agency)',
                'startup'          => 'موقع شركة ناشئة Startup',
                'classic_business' => 'موقع تعريفي (من نحن + خدماتنا + تواصل)',
            ],
        ],

        'lms' => [
            'label'       => 'مواقع كورسات وأكاديميات',
            'description' => 'منصات تعليمية، أكاديميات، مواقع مدربين وكورسات أونلاين.',
            'examples'    => [
                'موقع كورسات أونلاين (LMS)',
                'أكاديمية تدريب',
                'صفحة هبوط لكورس واحد أو برامج تدريبية',
                'موقع معلمين/مدربين أفراد',
            ],
            'subtypes'    => [
                'lms_platform'     => 'منصة كورسات أونلاين (LMS)',
                'academy'          => 'أكاديمية تدريب',
                'course_landing'   => 'صفحة هبوط لكورس واحد',
                'trainer_website'  => 'موقع مدرب أو معلم فردي',
            ],
        ],

        'store' => [
            'label'       => 'متاجر إلكترونية E-Commerce',
            'description' => 'متاجر رقمية أو فيزيكال، بيع منتجات وخدمات واشتراكات.',
            'examples'    => [
                'متجر منتجات فيزيكال (ملابس، أحذية، إكسسوارات، …)',
                'متجر منتجات رقمية (دورات، ملفات، Templates)',
                'متجر خدمات مدفوعة',
                'متجر اشتراكات (باقات شهرية/سنوية)',
            ],
            'subtypes'    => [
                'physical_store'   => 'متجر منتجات فيزيكال',
                'digital_store'    => 'متجر منتجات رقمية',
                'services_store'   => 'متجر خدمات مدفوعة',
                'subscription_shop'=> 'متجر اشتراكات (باقات)',
            ],
        ],

        'restaurant' => [
            'label'       => 'مطاعم وكافيهات',
            'description' => 'مواقع للمطاعم والكافيهات لعرض المنيو والحجوزات والطلبات.',
            'examples'    => [
                'منيو أونلاين',
                'موقع حجز طاولات',
                'موقع طلب أونلاين (لو ربطناه بـ WooCommerce)',
            ],
            'subtypes'    => [
                'online_menu'      => 'منيو أونلاين',
                'table_booking'    => 'موقع حجز طاولات',
                'online_orders'    => 'موقع طلب أونلاين',
            ],
        ],

        'realestate' => [
            'label'       => 'مواقع عقارات Real Estate',
            'description' => 'مخصصة للعقارات، المكاتب، الإيجار والبيع.',
            'examples'    => [
                'موقع عرض عقارات',
                'فلل/شقق/مكاتب للبيع أو الإيجار',
                'موقع مكتب تسويق عقاري',
            ],
            'subtypes'    => [
                'listing_site'     => 'موقع عرض عقارات',
                'units_listing'    => 'فلل/شقق/مكاتب للبيع أو الإيجار',
                'agency_realestate'=> 'موقع مكتب تسويق عقاري',
            ],
        ],

        'booking' => [
            'label'       => 'مواقع حجز وخدمات (Booking)',
            'description' => 'أي خدمة تعتمد على حجز موعد أو جلسة.',
            'examples'    => [
                'حجز مواعيد دكتور / عيادة',
                'صالون حلاقة / سبا / Beauty Center',
                'جيم / Fitness Center',
                'أي خدمة فيها Appointment / حجز وقت',
            ],
            'subtypes'    => [
                'doctor_clinic'    => 'حجز مواعيد دكتور / عيادة',
                'salon_spa'        => 'صالون حلاقة / سبا / Beauty Center',
                'gym_fitness'      => 'جيم / Fitness Center',
                'generic_booking'  => 'خدمة تعتمد على حجز وقت',
            ],
        ],

        'news' => [
            'label'       => 'مواقع أخبار ومجلات',
            'description' => 'مواقع محتوى إخباري أو تحليلي أو مجلات رقمية.',
            'examples'    => [
                'مجلة إلكترونية',
                'موقع أخبار عام',
                'موقع أخبار متخصص (تِك – رياضة – اقتصاد)',
            ],
            'subtypes'    => [
                'online_magazine'  => 'مجلة إلكترونية',
                'general_news'     => 'موقع أخبار عام',
                'niche_news'       => 'موقع أخبار متخصص',
            ],
        ],

        'ngo' => [
            'label'       => 'مواقع جمعيات ومنظمات',
            'description' => 'للجمعيات والمبادرات والمنظمات غير الربحية.',
            'examples'    => [
                'جمعيات خيرية',
                'مؤسسات غير ربحية',
                'مواقع تبرعات ودعم',
            ],
            'subtypes'    => [
                'charity'          => 'جمعية خيرية',
                'non_profit'       => 'مؤسسة غير ربحية',
                'donation_site'    => 'موقع تبرعات ودعم',
            ],
        ],

        'saas' => [
            'label'       => 'مواقع SaaS / شركات تقنية',
            'description' => 'منتجات رقمية، منصات SaaS، شركات تقنية وبرمجيات.',
            'examples'    => [
                'موقع منتج رقمي أو SaaS',
                'موقع شركة برمجيات',
                'صفحات تسعير Pricing + Features + Docs',
            ],
            'subtypes'    => [
                'saas_product'     => 'موقع منتج رقمي أو SaaS',
                'software_company' => 'موقع شركة برمجيات',
                'pricing_docs'     => 'موقع تسعير + خصائص + Docs',
            ],
        ],

        'landing' => [
            'label'       => 'Landing Pages',
            'description' => 'صفحات هبوط مركزة لحملات أو عروض أو منتج واحد.',
            'examples'    => [
                'صفحات هبوط لحملات إعلانية',
                'صفحات تسجيل في كورس أو خدمة',
                'صفحات عرض لمنتج واحد',
            ],
            'subtypes'    => [
                'campaign_landing' => 'صفحة هبوط لحملة إعلانية',
                'signup_landing'   => 'صفحة تسجيل في كورس أو خدمة',
                'single_product'   => 'صفحة عرض لمنتج واحد',
            ],
        ],

        'edu' => [
            'label'       => 'مواقع تعليمية بدون كورسات كاملة',
            'description' => 'تعريف بمؤسسات تعليمية بدون نظام كورسات كامل.',
            'examples'    => [
                'موقع مدرسة',
                'حضانة / Nursery',
                'مركز دروس خصوصية',
            ],
            'subtypes'    => [
                'school_site'      => 'موقع مدرسة',
                'nursery'          => 'حضانة / Nursery',
                'lessons_center'   => 'مركز دروس خصوصية',
            ],
        ],

        'team' => [
            'label'       => 'مواقع تيم / فريق / توظيف',
            'description' => 'عرض الفريق، الوظائف، وجمع السير الذاتية.',
            'examples'    => [
                'صفحة فريق العمل',
                'موقع لتجميع CV للفريق',
                'صفحة وظائف / Careers',
            ],
            'subtypes'    => [
                'team_page'        => 'صفحة فريق العمل',
                'cv_collection'    => 'موقع لتجميع CV للفريق',
                'careers_page'     => 'صفحة وظائف / Careers',
            ],
        ],

        'gallery' => [
            'label'       => 'مواقع جاليري ومعارض',
            'description' => 'مواقع تعتمد على الصور والعروض البصرية (معارض، Showrooms).',
            'examples'    => [
                'موقع مصوّر',
                'معرض أعمال فني',
                'مواقع Showroom للمنتجات',
            ],
            'subtypes'    => [
                'photographer_site' => 'موقع مصوّر',
                'art_gallery'       => 'معرض أعمال فني',
                'product_showroom'  => 'موقع Showroom للمنتجات',
            ],
        ],
    ];
}

/**
 * تحميل CSS/JS من ملفات خارجية لو موجودة
 */
add_action('admin_enqueue_scripts', 'packages_load_custom_assets');
function packages_load_custom_assets() {
    if (file_exists(PACKAGES_PLUGIN_DIR . 'style.css')) {
        echo '<style>';
        include PACKAGES_PLUGIN_DIR . 'style.css';
        echo '</style>';
    }

    if (file_exists(PACKAGES_PLUGIN_DIR . 'script.js')) {
        echo '<script>';
        include PACKAGES_PLUGIN_DIR . 'script.js';
        echo '</script>';
    }
}

/**
 * قوائم لوحة تحكم الشبكة
 */
add_action('network_admin_menu', 'packages_network_menus');
function packages_network_menus() {
    add_menu_page(
        'إدارة باقات المواقع',
        'الباقات',
        'manage_network',
        'packages-admin',
        'packages_admin_page',
        'dashicons-admin-generic',
        6
    );

    add_submenu_page(
        'packages-admin',
        'الباقات المفعلة',
        'الباقات المفعلة',
        'manage_network',
        'packages-active',
        'packages_active_page'
    );
}

/**
 * صفحة إدارة الباقات في الشبكة
 */
function packages_admin_page() {
    if (!current_user_can('manage_network')) {
        return;
    }

    $categories = get_site_option('packages_categories', []);

    $edit_mode     = false;
    $edit_id       = null;
    $edit_category = null;

    if (isset($_GET['edit'])) {
        $edit_id = intval($_GET['edit']);
        if (isset($categories[$edit_id])) {
            $edit_mode     = true;
            $edit_category = $categories[$edit_id];
        }
    }

    // حذف باقة
    if (isset($_GET['delete'])) {
        $delete_id = intval($_GET['delete']);
        if (isset($categories[$delete_id])) {
            unset($categories[$delete_id]);
            update_site_option('packages_categories', array_values($categories));
            echo '<div class="notice notice-success"><p>تم حذف الباقة بنجاح!</p></div>';
            $categories = get_site_option('packages_categories', []);
        }
    }

    // حفظ/تعديل باقة
    if (isset($_POST['save_category'])) {
        $category_name     = isset($_POST['category_name']) ? sanitize_text_field($_POST['category_name']) : '';
        $category_desc     = isset($_POST['category_desc']) ? sanitize_textarea_field($_POST['category_desc']) : '';
        $category_features = isset($_POST['category_features']) ? sanitize_textarea_field($_POST['category_features']) : '';
        $category_plugins  = isset($_POST['category_plugins']) ? array_map('sanitize_text_field', (array) $_POST['category_plugins']) : [];
        $category_theme    = isset($_POST['category_theme']) ? sanitize_text_field($_POST['category_theme']) : '';
        $category_type     = isset($_POST['category_type']) ? sanitize_text_field($_POST['category_type']) : 'personal';
        $category_subtype  = isset($_POST['category_subtype']) ? sanitize_text_field($_POST['category_subtype']) : '';

        $categories = get_site_option('packages_categories', []);

        if (!empty($_POST['edit_id']) && is_numeric($_POST['edit_id'])) {
            $id = intval($_POST['edit_id']);
            if (isset($categories[$id])) {
                $categories[$id] = [
                    'name'        => $category_name,
                    'desc'        => $category_desc,
                    'features'    => $category_features,
                    'plugins'     => $category_plugins,
                    'theme'       => $category_theme,
                    'site_type'   => $category_type,
                    'site_subtype'=> $category_subtype,
                ];
                echo '<div class="notice notice-success"><p>تم تعديل الباقة بنجاح!</p></div>';
            }
        } else {
            $categories[] = [
                'name'        => $category_name,
                'desc'        => $category_desc,
                'features'    => $category_features,
                'plugins'     => $category_plugins,
                'theme'       => $category_theme,
                'site_type'   => $category_type,
                'site_subtype'=> $category_subtype,
            ];
            echo '<div class="notice notice-success"><p>تم إضافة الباقة بنجاح!</p></div>';
        }

        update_site_option('packages_categories', $categories);
        $categories    = get_site_option('packages_categories', []);
        $edit_mode     = false;
        $edit_id       = null;
        $edit_category = null;
    }

    $themes     = wp_get_themes();
    $site_types = packages_get_site_types();

    echo '<div class="wrap packages-admin">';
    echo '<h1 class="page-title">' . ($edit_mode ? 'تعديل الباقة' : 'إضافة باقة جديدة') . '</h1>';

    $val_name      = $edit_mode ? $edit_category['name']        : '';
    $val_desc      = $edit_mode ? $edit_category['desc']        : '';
    $val_features  = $edit_mode ? $edit_category['features']    : '';
    $val_theme     = $edit_mode && isset($edit_category['theme']) ? $edit_category['theme'] : '';
    $val_type      = $edit_mode && isset($edit_category['site_type']) ? $edit_category['site_type'] : 'personal';
    $val_subtype   = $edit_mode && isset($edit_category['site_subtype']) ? $edit_category['site_subtype'] : '';

    echo '<div class="card">';
    echo '<form method="post">';
    echo '<input type="hidden" name="edit_id" value="' . ($edit_mode ? esc_attr($edit_id) : '') . '">';

    echo '<div class="form-group">
            <label for="category_name">اسم الباقة:</label>
            <input type="text" name="category_name" value="' . esc_attr($val_name) . '" required>
          </div>';

    echo '<div class="form-group">
            <label for="category_desc">وصف الباقة:</label>
            <textarea name="category_desc" required>' . esc_textarea($val_desc) . '</textarea>
          </div>';

    // نوع الموقع (تصنيف رئيسي)
    echo '<div class="form-group">
            <label for="category_type">نوع الموقع (التصنيف الرئيسي):</label>
            <select name="category_type" required>';
    foreach ($site_types as $key => $type_data) {
        echo '<option value="' . esc_attr($key) . '" ' . selected($val_type, $key, false) . '>'
            . esc_html($type_data['label']) . '</option>';
    }
    echo '  </select>
            <p style="font-size:12px;color:#777;margin-top:5px;">
                هذا يحدد القسم الذي ستظهر فيه الباقة للعميل (مثلاً: مواقع شخصية، متاجر إلكترونية، ...).
            </p>
          </div>';

    // نوع فرعي (داخل التصنيف)
    echo '<div class="form-group">
            <label for="category_subtype">النوع الفرعي داخل التصنيف:</label>
            <select name="category_subtype">
                <option value="">— بدون نوع فرعي محدد —</option>';
    foreach ($site_types as $t_key => $type_data) {
        if (empty($type_data['subtypes'])) {
            continue;
        }
        echo '<optgroup label="' . esc_attr($type_data['label']) . '">';
        foreach ($type_data['subtypes'] as $sub_key => $sub_label) {
            $selected = selected($val_subtype, $sub_key, false);
            echo '<option value="' . esc_attr($sub_key) . '" ' . $selected . '>' . esc_html($sub_label) . '</option>';
        }
        echo '</optgroup>';
    }
    echo '  </select>
            <p style="font-size:12px;color:#777;margin-top:5px;">
                مثال: موقع مصوّر، متجر منتجات رقمية، صفحة هبوط لحملة إعلانية... إلخ.
            </p>
          </div>';

    echo '<div class="form-group">
            <label for="category_features">ميزات الباقة (كل ميزة في سطر):</label>
            <textarea name="category_features" required>' . esc_textarea($val_features) . '</textarea>
          </div>';

    echo '<div class="form-group">
            <label for="category_theme">الثيم المرتبط بالباقة:</label>
            <select name="category_theme">
                <option value="">بدون تغيير الثيم</option>';
    foreach ($themes as $slug => $theme) {
        echo '<option value="' . esc_attr($slug) . '" ' . selected($val_theme, $slug, false) . '>'
            . esc_html($theme->get('Name')) . ' (' . esc_html($slug) . ')</option>';
    }
    echo '  </select>
          </div>';

    if (!function_exists('get_plugins')) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }
    $all_plugins      = get_plugins();
    $selected_plugins = $edit_mode && isset($edit_category['plugins']) ? (array) $edit_category['plugins'] : [];

    echo '<div class="form-group">
            <label>الإضافات المرتبطة بالباقة:</label><br>';
    foreach ($all_plugins as $plugin_file => $plugin_data) {
        echo '<div class="checkbox-item">
                <label>
                    <input type="checkbox" name="category_plugins[]" value="' . esc_attr($plugin_file) . '" ' . checked(in_array($plugin_file, $selected_plugins, true), true, false) . '>
                    ' . esc_html($plugin_data['Name']) . '
                </label>
              </div>';
    }
    echo '</div>';

    echo '<button type="submit" name="save_category" class="btn-primary">'
        . ($edit_mode ? 'تحديث الباقة' : 'حفظ الباقة') .
        '</button>';

    echo '</form>';
    echo '</div>';

    // جدول الباقات
    if (!empty($categories)) {
        echo '<h2 class="section-title">الباقات المحفوظة:</h2>';
        echo '<div class="table-responsive">';
        echo '<table class="packages-table">';
        echo '<thead>
                <tr>
                    <th>الاسم</th>
                    <th>الوصف</th>
                    <th>التصنيف</th>
                    <th>النوع الفرعي</th>
                    <th>الثيم</th>
                    <th>الإضافات</th>
                    <th>الإجراءات</th>
                </tr>
              </thead>';
        echo '<tbody>';

        foreach ($categories as $id => $category) {
            $type_key   = isset($category['site_type']) ? $category['site_type'] : 'personal';
            $type_label = isset($site_types[$type_key]) ? $site_types[$type_key]['label'] : 'غير محدد';

            $sub_label = '—';
            if (!empty($category['site_subtype'])) {
                foreach ($site_types as $tk => $tdata) {
                    if (!empty($tdata['subtypes']) && isset($tdata['subtypes'][$category['site_subtype']])) {
                        $sub_label = $tdata['subtypes'][$category['site_subtype']];
                        break;
                    }
                }
            }

            echo '<tr>';
            echo '<td>' . esc_html($category['name']) . '</td>';
            echo '<td>' . esc_html($category['desc']) . '</td>';
            echo '<td>' . esc_html($type_label) . '</td>';
            echo '<td>' . esc_html($sub_label) . '</td>';

            $theme_label = 'بدون ثيم محدد';
            if (!empty($category['theme'])) {
                $theme_obj = wp_get_theme($category['theme']);
                if ($theme_obj->exists()) {
                    $theme_label = $theme_obj->get('Name') . ' (' . $category['theme'] . ')';
                } else {
                    $theme_label = 'ثيم غير موجود: ' . esc_html($category['theme']);
                }
            }
            echo '<td>' . esc_html($theme_label) . '</td>';

            echo '<td>';
            if (!empty($category['plugins'])) {
                foreach ($category['plugins'] as $plugin) {
                    echo '<span class="badge">' . esc_html($plugin) . '</span> ';
                }
            }
            echo '</td>';

            echo '<td>
                    <a href="?page=packages-admin&edit=' . esc_attr($id) . '" class="btn-primary btn-small">تعديل</a>
                    <a href="?page=packages-admin&delete=' . esc_attr($id) . '" class="btn-danger btn-small" onclick="return confirm(\'هل أنت متأكد من الحذف؟\');">حذف</a>
                  </td>';

            echo '</tr>';
        }

        echo '</tbody>';
        echo '</table>';
        echo '</div>';
    }

    echo '</div>';
}

/**
 * صفحة الباقات المفعّلة على الشبكة
 */
function packages_active_page() {
    if (!current_user_can('manage_network')) {
        return;
    }

    $sites      = get_sites();
    $categories = get_site_option('packages_categories', []);

    echo '<div class="wrap packages-active">';
    echo '<h1 class="page-title">الباقات المفعلة على المواقع</h1>';
    echo '<div class="table-responsive">';
    echo '<table class="packages-table">';
    echo '<thead><tr><th>الموقع</th><th>الباقة المفعلة</th></tr></thead>';
    echo '<tbody>';

    foreach ($sites as $site) {
        switch_to_blog($site->blog_id);
        $active_package_id = get_option('active_package', -1);
        $package_name      = 'لا توجد باقة مفعلة';

        if ($active_package_id != -1 && isset($categories[$active_package_id])) {
            $package_name = $categories[$active_package_id]['name'];
        }

        echo '<tr><td>' . esc_html($site->blogname) . '</td><td>' . esc_html($package_name) . '</td></tr>';
        restore_current_blog();
    }

    echo '</tbody>';
    echo '</table>';
    echo '</div>';
    echo '</div>';
}

/**
 * قائمة الباقات في لوحة تحكم الموقع الفرعي
 */
add_action('admin_menu', 'packages_client_menu');
function packages_client_menu() {
    add_menu_page(
        'الباقات',
        'الباقات',
        'manage_options',
        'packages-client',
        'packages_client_page',
        'dashicons-screenoptions',
        6
    );
}

/**
 * صفحة الباقات في الموقع الفرعي (واجهة العميل)
 */
function packages_client_page() {
    if (!current_user_can('manage_options')) {
        return;
    }

    if (!function_exists('get_plugins')) {
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
    }

    $categories = get_site_option('packages_categories', []);
    $site_types = packages_get_site_types();
    $active_id  = get_option('active_package', -1);

    // تفعيل باقة
    if (isset($_POST['activate_package'])) {
        $package_id = intval($_POST['package_id']);
        if (isset($categories[$package_id])) {
            $selected_package = $categories[$package_id];
            $old_active_id    = get_option('active_package', -1);

            if ($old_active_id != -1 && isset($categories[$old_active_id])) {
                if (!empty($categories[$old_active_id]['plugins'])) {
                    foreach ($categories[$old_active_id]['plugins'] as $plugin) {
                        if (is_plugin_active($plugin)) {
                            deactivate_plugins($plugin);
                        }
                    }
                }
            }

            update_option('active_package', $package_id);

            if (!empty($selected_package['plugins'])) {
                foreach ($selected_package['plugins'] as $plugin) {
                    @activate_plugin($plugin);
                }
            }

            if (!empty($selected_package['theme'])) {
                $theme_slug = $selected_package['theme'];
                $theme_obj  = wp_get_theme($theme_slug);
                if ($theme_obj->exists()) {
                    switch_theme($theme_slug);
                }
            }

            echo '<div class="notice notice-success"><p>تم تفعيل الباقة بنجاح!</p></div>';
            $active_id = $package_id;
        }
    }

    // إلغاء تفعيل الباقة
    if (isset($_POST['deactivate_package'])) {
        if ($active_id != -1 && isset($categories[$active_id])) {
            if (!empty($categories[$active_id]['plugins'])) {
                foreach ($categories[$active_id]['plugins'] as $plugin) {
                    if (is_plugin_active($plugin)) {
                        deactivate_plugins($plugin);
                    }
                }
            }
            update_option('active_package', -1);
            $active_id = -1;
            echo '<div class="notice notice-success"><p>تم إلغاء تفعيل الباقة الحالية.</p></div>';
        }
    }

    // إعادة جلب
    $categories = get_site_option('packages_categories', []);
    $active_id  = get_option('active_package', -1);

    // تجميع حسب التصنيف
    $grouped = [];
    foreach ($categories as $id => $category) {
        $type_key = isset($category['site_type']) ? $category['site_type'] : 'personal';
        if (!isset($grouped[$type_key])) {
            $grouped[$type_key] = [];
        }
        $grouped[$type_key][$id] = $category;
    }

    echo '<div class="wrap packages-client sitycloud-packages-ui">';
    echo '<h1 class="page-title">اختر الباقة المناسبة لموقعك</h1>';
    echo '<p class="page-subtitle">كل نوع موقع له باقات جاهزة </p>';

    if (!empty($grouped)) {
        foreach ($site_types as $type_key => $type_data) {
            if (empty($grouped[$type_key])) {
                continue;
            }

            echo '<section class="packages-section">';
            echo '<div class="packages-section-header">';
            echo '<h2 class="packages-section-title">' . esc_html($type_data['label']) . '</h2>';
            if (!empty($type_data['description'])) {
                echo '<p class="packages-section-desc">' . esc_html($type_data['description']) . '</p>';
            }
            if (!empty($type_data['examples'])) {
                echo '<ul class="packages-section-examples">';
                foreach ($type_data['examples'] as $example) {
                    echo '<li>' . esc_html($example) . '</li>';
                }
                echo '</ul>';
            }
            echo '</div>';

            echo '<div class="packages-grid">';
            foreach ($grouped[$type_key] as $id => $category) {
                $is_active = ($active_id == $id);

                // اسم النوع الفرعي لو موجود
                $sub_label = '';
                if (!empty($category['site_subtype'])) {
                    foreach ($site_types as $tk => $tdata) {
                        if (!empty($tdata['subtypes']) && isset($tdata['subtypes'][$category['site_subtype']])) {
                            $sub_label = $tdata['subtypes'][$category['site_subtype']];
                            break;
                        }
                    }
                }

                echo '<div class="package-card' . ($is_active ? ' package-card--active' : '') . '">';
                echo '<div class="package-card-header">';
                echo '<h3 class="package-title">' . esc_html($category['name']) . '</h3>';
                if ($is_active) {
                    echo '<span class="package-badge-active">الباقة الحالية</span>';
                }
                echo '</div>';

                if (!empty($sub_label)) {
                    echo '<div class="package-subtype-pill">نوع الموقع: ' . esc_html($sub_label) . '</div>';
                }

                echo '<p class="package-desc">' . esc_html($category['desc']) . '</p>';

                echo '<ul class="features-list">';
                $features = explode("\n", $category['features']);
                foreach ($features as $feature) {
                    $feature = trim($feature);
                    if ($feature !== '') {
                        echo '<li>' . esc_html($feature) . '</li>';
                    }
                }
                echo '</ul>';

               

                echo '<form method="post" class="package-form">';
                echo '<input type="hidden" name="package_id" value="' . esc_attr($id) . '">';
                echo '<button type="submit" name="activate_package" class="btn-primary">'
                    . ($is_active ? 'إعادة تفعيل الباقة' : 'تفعيل هذه الباقة') .
                    '</button>';
                echo '</form>';

                echo '</div>';
            }
            echo '</div>'; // grid
            echo '</section>';
        }
    } else {
        echo '<p class="no-packages">لا توجد باقات متاحة حاليًا. تواصل مع دعم سايتي كلاود لتفعيل الباقات.</p>';
    }

    if ($active_id != -1) {
        echo '<form method="post" class="deactivate-form">';
        echo '<button type="submit" name="deactivate_package" class="btn-danger">إلغاء تفعيل الباقة الحالية</button>';
        echo '</form>';
    }

    echo '</div>';
}

/**
 * CSS واجهة مودرن (أبيض + بنفسجي + زجاجي)
 */
add_action('admin_head', 'packages_include_css');
function packages_include_css() {
    echo '<style>
    .wrap.sitycloud-packages-ui {
        max-width: 1400px;
    }
    .sitycloud-packages-ui {
        font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
    }
    .sitycloud-packages-ui .page-title {
        font-size: 26px;
        font-weight: 700;
        margin-bottom: 8px;
        color: #1f1633;
    }
    .sitycloud-packages-ui .page-subtitle {
        font-size: 13px;
        color: #6c6c80;
        margin-bottom: 22px;
    }
    .packages-section {
        margin-bottom: 32px;
        padding: 16px 18px 20px;
        border-radius: 18px;
        background: radial-gradient(circle at top left, rgba(124,106,255,0.2), rgba(255,255,255,0.9));
        border: 1px solid rgba(124,106,255,0.35);
    }
    .packages-section-header {
        margin-bottom: 14px;
    }
    .packages-section-title {
        font-size: 18px;
        font-weight: 700;
        margin: 0 0 4px;
        color: #251b4b;
    }
    .packages-section-desc {
        font-size: 13px;
        color: #5b5670;
        margin: 0 0 8px;
    }
    .packages-section-examples {
        display: flex;
        flex-wrap: wrap;
        gap: 6px;
        margin: 0;
        padding: 0;
        list-style: none;
    }
    .packages-section-examples li {
        font-size: 11px;
        padding: 4px 9px;
        border-radius: 999px;
        background: rgba(255,255,255,0.9);
        border: 1px solid rgba(124,106,255,0.2);
        color: #403465;
    }

    .packages-grid {
        display: grid;
        grid-template-columns: repeat(auto-fill, minmax(260px, 1fr));
        gap: 15px;
        margin-top: 12px;
    }
    .package-card {
        position: relative;
        border-radius: 16px;
        padding: 16px 16px 14px;
        background: rgba(255,255,255,0.95);
        backdrop-filter: blur(12px);
        box-shadow: 0 10px 24px rgba(15,11,43,0.08);
        border: 1px solid rgba(230,227,255,0.9);
        display: flex;
        flex-direction: column;
        transition: transform 0.08s ease-out, box-shadow 0.08s ease-out, border-color 0.08s;
    }
    .package-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 16px 30px rgba(15,11,43,0.15);
        border-color: rgba(124,106,255,0.7);
    }
    .package-card--active {
        border: 1px solid rgba(91,63,211,0.95);
        box-shadow: 0 0 0 1px rgba(91,63,211,0.35), 0 16px 36px rgba(91,63,211,0.35);
    }
    .package-card-header {
        display: flex;
        align-items: center;
        justify-content: space-between;
        gap: 8px;
        margin-bottom: 6px;
    }
    .package-title {
        font-size: 16px;
        font-weight: 700;
        margin: 0;
        color: #251b4b;
    }
    .package-badge-active {
        font-size: 11px;
        padding: 4px 9px;
        border-radius: 999px;
        background: rgba(91,63,211,0.12);
        color: #5b3fd3;
        font-weight: 600;
    }
    .package-subtype-pill {
        display: inline-block;
        margin-bottom: 8px;
        font-size: 11px;
        padding: 3px 9px;
        border-radius: 999px;
        background: rgba(124,106,255,0.08);
        color: #3a2a9c;
        border: 1px solid rgba(124,106,255,0.3);
    }
    .package-desc {
        font-size: 13px;
        color: #5f5a7c;
        margin: 0 0 8px;
    }
    .features-list {
        margin: 0 0 8px;
        padding-left: 18px;
        font-size: 12px;
        color: #4a4665;
    }
    .features-list li {
        margin-bottom: 3px;
    }
    .package-theme {
        font-size: 12px;
        color: #4a4665;
        margin: 0 0 6px;
    }
    .package-plugins-title {
        font-size: 12px;
        color: #5b5670;
        margin: 0 0 4px;
    }
    .package-plugins {
        margin-bottom: 10px;
    }
    .badge {
        background: rgba(91,63,211,0.12);
        color: #3a2a9c;
        padding: 3px 8px;
        border-radius: 999px;
        font-size: 11px;
        display: inline-block;
        margin: 0 4px 4px 0;
        border: 1px solid rgba(91,63,211,0.3);
    }
    .btn-primary {
        background: linear-gradient(135deg,#5b3fd3,#7c68ff);
        color: #fff;
        border: none;
        padding: 9px 16px;
        border-radius: 999px;
        cursor: pointer;
        font-size: 13px;
        font-weight: 600;
        box-shadow: 0 10px 20px rgba(91,63,211,0.25);
        transition: transform 0.07s ease, box-shadow 0.07s ease, background 0.12s ease;
        align-self: flex-start;
    }
    .btn-primary:hover {
        transform: translateY(-1px);
        box-shadow: 0 14px 24px rgba(91,63,211,0.35);
    }
    .btn-primary.btn-small {
        padding: 6px 12px;
        font-size: 12px;
        box-shadow: none;
    }
    .btn-danger {
        background: #e5425f;
        color: #fff;
        border: none;
        padding: 8px 14px;
        border-radius: 999px;
        cursor: pointer;
        font-size: 12px;
        font-weight: 500;
    }
    .deactivate-form {
        margin-top: 18px;
    }

    .packages-table {
        width: 100%;
        border-collapse: collapse;
        margin-top: 12px;
        background: #fff;
        border-radius: 12px;
        overflow: hidden;
    }
    .packages-table th,
    .packages-table td {
        padding: 9px 10px;
        border: 1px solid #eee;
        font-size: 13px;
        vertical-align: top;
    }
    .packages-table th {
        background: #f7f6ff;
        font-weight: 600;
        color: #3a3165;
    }

    .card {
        background: #fff;
        padding: 18px;
        border-radius: 14px;
        box-shadow: 0 8px 18px rgba(15,11,43,0.06);
        margin-bottom: 20px;
        border: 1px solid #eeeafb;
    }
    .form-group {
        margin-bottom: 14px;
    }
    .form-group label {
        display: block;
        font-weight: 600;
        font-size: 13px;
        margin-bottom: 4px;
        color: #282244;
    }
    .form-group input[type="text"],
    .form-group textarea,
    .form-group select {
        width: 100%;
        padding: 7px 9px;
        border: 1px solid #ddd;
        border-radius: 8px;
        box-sizing: border-box;
        font-size: 13px;
    }
    .checkbox-item {
        margin-bottom: 6px;
        font-size: 12px;
    }
    .checkbox-item input {
        margin-right: 4px;
    }

    .no-packages {
        font-size: 13px;
        color: #777;
        font-style: italic;
    }

    @media (max-width: 782px) {
        .packages-grid {
            grid-template-columns: 1fr;
        }
        .packages-section {
            padding: 12px 10px 14px;
        }
    }
    </style>';
}

/**
 * JS بسيط
 */
add_action('admin_footer', 'packages_include_js');
function packages_include_js() {
    echo '<script>
    jQuery(document).ready(function($) {
        console.log("Sity Cloud Packages UI loaded ✅");
    });
    </script>';
}
